import React, { useState, useEffect } from "react";
import "./App.css";
import {
  StarPrinterFactory,
  InterfaceType,
  StarConnectionSettings,
  Options,
  StarXpandCommand,
  StarIO10Logger,
} from "star-io10-web";

function Printing() {
  const [identifier, setIdentifier] = useState<string>("");
  const [printerFactory, setPrinterFactory] = useState<StarPrinterFactory>(
    () => {
      console.log("create printer factory");
      return new StarPrinterFactory();
    },
  );
  const handleBeforeUnloadEvent = (event: BeforeUnloadEvent) => {
    showMessage("dispose printer factory");
    printerFactory.dispose();
  };
  useEffect(() => {
    // when the window is closed, dispose the printer factory.
    window.addEventListener("beforeunload", handleBeforeUnloadEvent, true);
    return () =>
      window.removeEventListener("beforeunload", handleBeforeUnloadEvent, true);
  }, []);

  const [isProcessing, setIsProcessing] = useState<boolean>(false);
  const [message, setMessage] = useState<string>("");
  async function showMessage(msg: string) {
    console.log(msg);
    setMessage((prev: string) => prev + msg + "\n");
  }

  async function onPressPrintButton() {
    showMessage("on press print button");
    setIsProcessing(true);

    // Start debug log output to console
    // StarIO10Logger.start();

    const connectionSettings = new StarConnectionSettings();
    connectionSettings.interfaceType = InterfaceType.Usb;
    if (identifier.length > 0) {
      connectionSettings.identifier = identifier;
    }

    showMessage("create StarPrinter ");
    const printer = printerFactory.createStarPrinter(connectionSettings);
    // const printer = new StarPrinter(connectionSettings);
    // `new StarPrinter(connectionSettings)` may take a long time to process.
    // If you want to separate the process to create StarPrinter, use StarPrinterFactory.

    try {
      await printer.open();
      showMessage("open success");

      let commands = await createPeripheralCommands();
      const options = new Options.PrintOptions();
      options.enableCheckStatus = false;
      await printer.print(commands, options);

      commands = await createPrintCommands();
      await printer.print(commands);

      showMessage("print success");

      await printer.close();
      showMessage("close success");
    } catch (error) {
      try {
        await printer?.close();
      } catch {
        // do nothing.
      }

      showMessage("print failed " + error);
      console.error(error);
    } finally {
      await printer.dispose();
      // StarIO10Logger.stop();
      setIsProcessing(false);
    }
  }

  async function createPeripheralCommands(): Promise<string> {
    const builder = new StarXpandCommand.StarXpandCommandBuilder();
    builder.addDocument(
      new StarXpandCommand.DocumentBuilder()

        .addDrawer(
          new StarXpandCommand.DrawerBuilder().actionOpen(
            new StarXpandCommand.Drawer.OpenParameter().setChannel(
              StarXpandCommand.Drawer.Channel.No1,
            ),
          ),
        )

        .addMelodySpeaker(
          new StarXpandCommand.MelodySpeakerBuilder().actionDriveOneTimeSound(
            new StarXpandCommand.MelodySpeaker.DriveOneTimeSoundParameter(
              "sound0.wav",
            ).setVolume(5),
          ),
        )

        .addDisplay(
          new StarXpandCommand.DisplayBuilder()
            .actionClearAll()
            .actionShowImage(
              new StarXpandCommand.Display.ImageParameter("logo_01.png"),
            ),
        ),
    );

    const command = await builder.getCommands();
    return Promise.resolve(command);
  }

  async function createPrintCommands(): Promise<string> {
    const builder = new StarXpandCommand.StarXpandCommandBuilder();
    builder.addDocument(
      new StarXpandCommand.DocumentBuilder().addPrinter(
        new StarXpandCommand.PrinterBuilder()
          .actionPrintImage(
            new StarXpandCommand.Printer.ImageParameter("logo_01.png", 406),
          )
          .styleInternationalCharacter(
            StarXpandCommand.Printer.InternationalCharacterType.Usa,
          )
          .styleCharacterSpace(0)
          .styleAlignment(StarXpandCommand.Printer.Alignment.Center)
          .actionPrintText(
`Star Clothing Boutique
123 Star Road
City, State 12345

`,
          )
          .styleAlignment(StarXpandCommand.Printer.Alignment.Left)
          .actionPrintText(
`Date:MM/DD/YYYY    Time:HH:MM PM
--------------------------------

`,
          )
          .actionPrintText(
`SKU         Description    Total
300678566   PLAIN T-SHIRT  10.99
300692003   BLACK DENIM    29.99
300651148   BLUE DENIM     29.99
300642980   STRIPED DRESS  49.99
300638471   BLACK BOOTS    35.99

Subtotal                  156.95
Tax                         0.00
--------------------------------
`,
          )
          .actionPrintText(
`Total `
          )
          .add(
            new StarXpandCommand.PrinterBuilder()
              .styleMagnification(
                new StarXpandCommand.MagnificationParameter(2, 2),
              )
              .actionPrintText(
`   $156.95
`,
              ),
          )
          .actionPrintText(
`--------------------------------

Charge
156.95
Visa XXXX-XXXX-XXXX-0123

`,
          )
          .add(
            new StarXpandCommand.PrinterBuilder()
              .styleInvert(true)
              .actionPrintText(
`Refunds and Exchanges
`,
              ),
          )
          .actionPrintText(
`Within `
          )
          .add(
            new StarXpandCommand.PrinterBuilder()
              .styleUnderLine(true)
              .actionPrintText(
`30 days`
              ),
          )
          .actionPrintText(
` with receipt
`,
          )
          .actionPrintText(
`And tags attached

`,
          )
          .styleAlignment(StarXpandCommand.Printer.Alignment.Center)
          .actionPrintBarcode(
            new StarXpandCommand.Printer.BarcodeParameter(
              "0123456",
              StarXpandCommand.Printer.BarcodeSymbology.Jan8,
            )
              .setBarDots(3)
              .setBarRatioLevel(
                StarXpandCommand.Printer.BarcodeBarRatioLevel.Level0,
              )
              .setHeight(5)
              .setPrintHri(true),
          )
          .actionFeedLine(1)
          .actionPrintQRCode(
            new StarXpandCommand.Printer.QRCodeParameter("Hello World.\n")
              .setModel(StarXpandCommand.Printer.QRCodeModel.Model2)
              .setLevel(StarXpandCommand.Printer.QRCodeLevel.L)
              .setCellSize(8),
          )
          .actionCut(StarXpandCommand.Printer.CutType.Partial),
      ),
    );

    const commands = await builder.getCommands();
    return Promise.resolve(commands);
  }

  return (
    <div className="Printing">
      {isProcessing && (
        <div className="App-overlay">
          <div className="App-overlay-message">Printing...</div>
        </div>
      )}
      <div>
        <label>Identifier: </label>
        <input
          value={identifier}
          onChange={(e) => setIdentifier(e.target.value)}
        />
        <br />
        <button id="print" onClick={onPressPrintButton}>
          Print
        </button>
        <br />
        <div style={{ whiteSpace: "pre-line" }}>
          <p>{message}</p>
        </div>
      </div>
    </div>
  );
}

export default Printing;
